//
//  Puzzle.swift
//  15-puzzle
//
//  Created by Илья Лошкарёв on 19.02.17.
//  Copyright © 2017 Илья Лошкарёв. All rights reserved.
//

import Foundation

protocol PuzzleDelegate: class { // weak delegates work only for class-bound protocols
    func puzzleDidSwitchTiles(puzzle: Puzzle, firstTilePos: (Int,Int), secondTilePos: (Int,Int))
    func puzzleIsSolved(puzzle: Puzzle)
}

class Puzzle {
    let size: Int
    var board: [[Int]]
    weak var delegate: PuzzleDelegate?  // break potencial cycle
    
    init(of size: Int) {
        self.size = size
        board = [[Int]]()
        for i in 0..<size {
            board.append([Int]((size*i+1)...(size*(i+1))))
        }
        board[size - 1][size - 1] = 0
        shuffle()
    }
    
    func shuffle() {
        let offsets = [(1,0),(0,1),(-1,0),(0,-1)]
        var (i,j) = blankPosition
        for _ in 1...size * size {
            let t = Int.random(in: 0..<offsets.count)
            let k = (i + offsets[t].0) < 0 ? 0 :
                (i + offsets[t].0) == size ? size-1 : (i + offsets[t].0)
            let l = (j + offsets[t].1) < 0 ? 0 :
                (j + offsets[t].1) >= size ? size-1 : (j + offsets[t].1)
            if i != k || j != l {
                (board[k][l], board[i][j]) = (board[i][j], board[k][l])
                i = k
                j = l
                //print(i,j,"->",k,l)
            }
        }
    }
    
    var blankPosition: (Int,Int) {
        return positionOf(number: 0)!
    }
    
    func positionOf(number: Int) -> (Int,Int)? {
        guard (0..<size * size).contains(number) else {
            return nil
        }
        let i = board.firstIndex(where: { $0.contains(number)})!
        return (i, board[i].firstIndex(where: { $0 == number })!)
    }
    
    func hasNumberAdjancedToBlank(_ number: Int) -> Bool {
        guard let (i, j) = positionOf(number: number) else {
            return false
        }
        let (blankRow, blankCol) = blankPosition
        
        return
            i == blankRow + 1 && j == blankCol ||
            i == blankRow - 1 && j == blankCol ||
            i == blankRow && j == blankCol + 1 ||
            i == blankRow && j == blankCol - 1 
    }
    
    func switchBlank(with tileNumber: Int){
        let (i, j) = positionOf(number: tileNumber)!
        let (blankRow, blankCol) = blankPosition
        (board[i][j], board[blankRow][blankCol]) = (board[blankRow][blankCol], board[i][j])
        
        delegate?.puzzleDidSwitchTiles(puzzle: self, firstTilePos: (i,j), secondTilePos: (blankRow, blankCol))
        
        if solved {
            delegate?.puzzleIsSolved(puzzle: self)
        }
    }
    
    var solved: Bool {
        if board[size - 1][size - 1] != 0 { return false }
        for i in  0..<size {
            for j in 0..<size {
                if i == size - 1 && j == size - 1 && board[i][j] == 0 {
                    return true
                } else if board[i][j] != i * size + j + 1 {
                    return false
                }
            }
        }
        return true
    }

}
